#ifndef PLUGININTERFACE_H
#define PLUGININTERFACE_H

#include <QtPlugin>
#include <QString>
#include <QTableWidget>

#include "global.h"
#include "common.h"

struct CoreInterface;

/*!
    \brief Base class for all TuPP plugins

    Most of TuPP's functionality is implemented in plugins.
    All plugins derive from TuppPlugin.

    Upon compilation, the binary file (.dll or .so) will be created in the TuPP plugins directory.
    At startup, TuPP will load all plugins it finds in this directory, and initialize them.

    In the \c initialize() function, the plugin should get a copy of the the \l{CoreInterface} pointer passed as parameter.
    To see examples of this, refer to the system plugins. One simple example is in plugins/template.

    Plugins can communicate with one another via \l{SignalManager}.

    See \l{CoreInterface} for more information about the core classes of TuPP.
*/

class CORE_INTERFACE_EXPORT PluginInterface
{
public:
    virtual ~PluginInterface();

    ///Plugs and initializes the plugin
    virtual void plug(CoreInterface* core);
    /*!Makes the plugin active.
       Returns true if plugin was successfully activated.
       Each programmer plugin should call \c restoreSettings() in this method because
       global restore action is done for active plugins only.*/
    virtual bool activate();
    /*!Makes the programmer plugin inactive (\c isProgrammerPlugin() returns true).
       Returns true if plugin was successfully deactivated.
       Each programmer plugin should call \c saveSettings() in this method because
       global save action is done for active plugins only.*/
    virtual bool deactivate();

    ///Name of the plugin
    virtual QString name() const;
    ///Version of the plugin
    virtual QString version() const;
    ///Description of the plugin
    virtual QString description() const;
    ///Authors - may be in rich format
    virtual QString authors() const;
    ///A number affecting when the plugin is loaded relative to other plugins.
    ///Plugins with lower load orders are loaded first.
    ///Programmer plugins have to have non-negative value.
    virtual int loadOrder() const;
    ///Returns true if this plugin is programmer plugin (can be unplugged (means deactivated) during run-time)
    virtual bool isProgrammerPlugin() const;

    /*!If \c isProgrammerPlugin() returns true the plugin \a P should fill
    the table of supported chips \a chipsTable. The argument \a notes can
    contain notes under the table in case the parameter \a notes isn't a NULL pointer.
    Method returns true if the table was successfully filled.*/
    virtual bool supportedChipsTable(QTableWidget* chipsTable, QString* notes = NULL);

    ///Shows plugin's settings dialog
    virtual void settings();

    ///Restores plugin specific settings.
    ///For programmer plugins is this method called only if plugin is active.
    virtual void restoreSettings();
    ///Saves plugin specific settings.
    ///For programmer plugins is this method called only if plugin is active.
    virtual void saveSettings();
};

//If the version is changed, update the version also in file "tuppcoreinterface.pro" and "common.h"
Q_DECLARE_INTERFACE(PluginInterface, "TuPP.PluginInterface/0.0.1")

#endif //PLUGININTERFACE_H
